<?php
namespace App\Http\Controllers\MobileApp;

use App\Models\SparePart;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Cache;
use App\Http\Resources\SparePartResource;

class SparePartController extends Controller
{
    /**
     * Constructor to set locale
     */
    public function __construct(Request $request)
    {
        $locale = $request->header('Accept-Language', 'en');
        \Illuminate\Support\Facades\App::setLocale($locale);
    }

    /**
     * Get all spare parts
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $locale = \Illuminate\Support\Facades\App::getLocale();
            $limit = $request->query('limit', 10);
            $page = $request->query('page', 1);
            $search = $request->query('search', '');
            $model_id = $request->query('model_id');
            $category_id = $request->query('category_id');
            $vendor_id = $request->query('vendor_id');
            $city_id = $request->query('city_id');

            $limit = max(1, min(100, (int) $limit));

            $cacheKey = 'spare_parts_' . $locale . '_limit_' . $limit . '_page_' . $page . '_search_' . md5($search)
                . '_model_' . ($model_id ?? 'none')
                . '_category_' . ($category_id ?? 'none')
                . '_vendor_' . ($vendor_id ?? 'none')
                . '_city_' . ($city_id ?? 'none');

            $spareParts = Cache::remember($cacheKey, 3600, function () use ($limit, $search, $model_id, $category_id, $vendor_id, $city_id) {
                $query = SparePart::with(['category', 'vendor', 'model', 'city'])->where('is_active', true);

                if (!empty($search)) {
                    $query->where(function ($q) use ($search) {
                        $q->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(name, '$.en')) LIKE ?", ['%' . $search . '%'])
                            ->orWhereRaw("JSON_UNQUOTE(JSON_EXTRACT(name, '$.ar')) LIKE ?", ['%' . $search . '%'])
                            ->orWhere('description', 'LIKE', '%' . $search . '%');
                    });
                }

                if (!is_null($model_id)) {
                    $query->where('model_id', $model_id);
                }

                if (!is_null($category_id)) {
                    $query->where('category_id', $category_id);
                }

                if (!is_null($vendor_id)) {
                    $query->where('vendor_id', $vendor_id);
                }

                if (!is_null($city_id)) {
                    $query->where('city_id', $city_id);
                }

                return $query->orderBy('id')->paginate($limit);
            });

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.spare_parts_retrieved'),
                'data' => SparePartResource::collection($spareParts),
                'pagination' => [
                    'current_page' => $spareParts->currentPage(),
                    'last_page' => $spareParts->lastPage(),
                    'per_page' => $spareParts->perPage(),
                    'total' => $spareParts->total(),
                    'next_page_url' => $spareParts->nextPageUrl(),
                    'prev_page_url' => $spareParts->previousPageUrl(),
                ],
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error retrieving spare parts: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_retrieve_spare_parts'),
            ], 500);
        }
    }


    /**
     * Get a single spare part
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(Request $request, $id)
    {
        try {
            $locale = \Illuminate\Support\Facades\App::getLocale();

            $sparePart = Cache::remember('spare_part_' . $id . '_' . $locale, 3600, function () use ($id) {
                return SparePart::with(['category', 'vendor'])
                    ->where('is_active', true)
                    ->findOrFail($id);
            });

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.spare_part_retrieved'),
                'data' => new SparePartResource($sparePart),
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.spare_part_not_found'),
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error retrieving spare part: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_retrieve_spare_part'),
                // Remove error message in production for security
                // 'error' => $e->getMessage(),
            ], 500);
        }
    }
    public function data()
    {

    }
}