<?php
namespace App\Http\Controllers\MobileApp;

use Carbon\Carbon;
use App\Models\User;
use App\Mail\PasswordReset;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Http\Resources\UserResource;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    /**
     * Constructor to set the locale based on Accept-Language header
     */
    public function __construct(Request $request)
    {
        $locale = $request->header('Accept-Language', 'ar');
        App::setLocale($locale);
    }

    /**
     * Register a new user
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'full_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20|unique:users',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'city_id' => 'required|exists:cities,id',
            'device_token' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = User::create([
            'full_name' => $request->full_name,
            'phone' => $request->phone,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'city_id' => $request->city_id,
            'email_verification_code' => Str::random(40),
            'email_verified' => false,
            'device_token' => $request->device_token, // Save device token
        ]);

        $token = $user->createToken('mobile-app')->plainTextToken;

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.user_registered'),
            'user' => $user,
            'token' => $token,
        ], 201);
    }
    /**
     * Login user and issue token
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email',
            'password' => 'required|string',
            'device_token' => 'required|string', // Add device token validation
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        if (!Auth::attempt($request->only('email', 'password'))) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.invalid_credentials'),
            ], 401);
        }

        $user = Auth::user();
        $user->update(['device_token' => $request->device_token]); // Save device token
        $token = $user->createToken('mobile-app')->plainTextToken;

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.user_logged_in'),
            'user' => $user,
            'token' => $token,
        ], 200);
    }

    /**
     * Logout user and revoke token
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout(Request $request)
    {
        $user = $request->user();

        if ($user && $user->currentAccessToken()) {
            $user->currentAccessToken()->delete();
        }

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.user_logged_out'),
        ], 200);
    }


    /**
     * Send password reset link/code
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function forgotPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = User::where('email', $request->email)->first();
        $token = mt_rand(100000, 999999);

        DB::table('password_reset_tokens')->updateOrInsert(
            ['email' => $request->email],
            [
                'token' => $token,
                'created_at' => Carbon::now(),
            ]
        );

        $resetLink = env('FRONTEND_URL', 'http://localhost') . '/reset-password?token=' . $token . '&email=' . urlencode($request->email);

        Mail::to($user->email)->send(new PasswordReset($resetLink, $user));

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.password_reset_link_sent'),
            'token' => $token

        ], 200);
    }

    /**
     * Reset password using token
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|exists:users,email',
            'token' => 'required|string',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $passwordReset = DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->where('token', $request->token)
            ->first();

        if (!$passwordReset || Carbon::parse($passwordReset->created_at)->addHour()->isPast()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.invalid_token'),
            ], 400);
        }

        $user = User::where('email', $request->email)->first();
        $user->update([
            'password' => Hash::make($request->password),
        ]);

        DB::table('password_reset_tokens')->where('email', $request->email)->delete();

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.password_reset_success'),
        ], 200);
    }

    /**
     * Save device token for push notifications
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function saveDeviceToken(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'token' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = Auth::user();
        $user->update(['device_token' => $request->token]);

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.device_token_saved'),
        ], 200);
    }

    /**
     * Update user profile
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateProfile(Request $request)
    {
        try {
            $user = auth('sanctum')->user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.unauthorized'),
                ], 401);
            }

            $validator = Validator::make($request->all(), [
                'full_name' => 'sometimes|string|max:255',
                'phone' => ['sometimes', 'string', 'max:20', 'unique:users,phone,' . $user->id],
                'email' => ['sometimes', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
                'city_id' => 'sometimes|exists:cities,id',
                'image' => 'sometimes|image|mimes:jpg,png,jpeg|max:2048',
                'device_token' => 'sometimes|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.validation_failed'),
                    'errors' => $validator->errors(),
                ], 422);
            }

            $data = $validator->validated();

            // Handle image upload
            if ($request->hasFile('image')) {
                $data['image'] = $request->file('image')->store('profiles', 'public');
            }

            DB::beginTransaction();

            $user->update($data);

            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.profile_updated'),
                'data' => new UserResource($user)
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating user profile', [
                'user_id' => $user->id ?? null,
                'data' => $request->all(),
                'locale' => app()->getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_update_profile'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }
    public function userData()
    {
        return response()->json([
            'status' => 'success',
            'message' => trans('messages.profile_updated'),
            'data' => new UserResource(auth('sanctum')->user())
        ], 200);
    }
    public function deleteAccount(Request $request)
    {
        try {
            $user = auth('sanctum')->user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.unauthorized'),
                ], 401);
            }

            // Log deletion attempt
            Log::info('Account deletion attempt', [
                'user_id' => $user->id,
                'email' => $user->email,
                'locale' => app()->getLocale(),
            ]);

            DB::transaction(function () use ($user) {
                // Soft delete related records
                $user->userCars()->delete();
                //  $user->carsForSale()->delete();
                //  $user->customPartRequests()->delete();
                $user->carts()->delete();
                $user->orders()->delete();
                $user->towRequests()->delete();
                $user->notifications()->delete();

                // Revoke all tokens
                $user->tokens()->delete();

                // Soft delete the user
                $user->delete();
            });

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.account_deleted'),
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error deleting user account', [
                'user_id' => auth('sanctum')->user()->id ?? null,
                'email' => auth('sanctum')->user()->email ?? null,
                'locale' => app()->getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_delete_account'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }
}